﻿Imports System.IO.Ports
Imports System.Text.RegularExpressions

Public Class FM302


#Region "Enums"

    Enum KeyState As Integer
        KeyboardLocked
        KeyboardUnlocked
    End Enum

    Enum SoundState As Integer
        SoundOn
        SoundOff
    End Enum

    Enum CouplingState As Integer
        CouplingDC
        CouplingAC
    End Enum

    Enum ModeState As Integer
        NotSet
        Absolute
        Relative
        Minimum
        Maximum
        AbsMax
    End Enum

    Enum GainState As Integer
        GainX1
        GainX10
        GainX100
    End Enum

    Enum UnitState As Integer
        UnitNotSet
        UnitTesla
        UnitGauss
        UnitOersted
        UnitA_per_m
        UnitA_per_cm
    End Enum

#End Region

#Region "Structs"

    Public Structure MeasuredValueStruct
        Public Value As String
        Public Unit As String
    End Structure

    Private Structure _VersionStruct
        Dim Value As String
        Dim Major As Integer
        Dim Minor As Integer
    End Structure

#End Region

#Region "Private Vars"

    Private WithEvents SerialPort As RS232

    Private _AbsMax As MeasuredValueStruct
    Private _Coupling As CouplingState
    Private _Digits As Integer
    Private _Filter As Integer
    Private _Gain As GainState
    Private _KeyLock As KeyState
    Private _Maximum As MeasuredValueStruct
    Private _MeasuredValue As MeasuredValueStruct
    Private _Minimum As MeasuredValueStruct
    Private _Mode As ModeState = ModeState.NotSet
    Private _NoProbe As Boolean
    Private _Overload As Boolean
    Private _Range As MeasuredValueStruct
    Private _Reference As MeasuredValueStruct
    Private _Serial As String
    Private _Sound As SoundState
    Private _Time As Integer
    Private _Unit As UnitState = UnitState.UnitNotSet
    Private _Version As _VersionStruct
    Private _Zero As Integer

    Private _ModeList As New Dictionary(Of ModeState, String)
    Private _UnitLIst As New Dictionary(Of UnitState, String)

#End Region

#Region "Public Events"

    Public Event NewAbsMax As EventHandler
    Public Event NewCoupling As EventHandler
    Public Event NewDigits As EventHandler
    Public Event NewFilter As EventHandler
    Public Event NewGain As EventHandler
    Public Event NewKeyLock As EventHandler
    Public Event NewMaximum As EventHandler
    Public Event NewMeasuredValue As EventHandler
    Public Event NewMinimum As EventHandler
    Public Event NewMode As EventHandler
    Public Event NewModes As EventHandler
    Public Event NewNoProbe As EventHandler
    Public Event NewOverload As EventHandler
    Public Event NewRange As EventHandler
    Public Event NewReference As EventHandler
    Public Event NewSerial As EventHandler
    Public Event NewSound As EventHandler
    Public Event NewTime As EventHandler
    Public Event NewUnit As EventHandler
    Public Event NewUnits As EventHandler
    Public Event NewVersion As EventHandler
    Public Event NewZero As EventHandler

    Public Event ZeroOutOfRange As EventHandler

#End Region

#Region "Public Methods"

    Public Sub New()
        SerialPort = New RS232

        'make settings for serial port
        With SerialPort
            .BaudRate = 9600
            .DataBits = 8
            .Handshake = IO.Ports.Handshake.None
            .NewLine = vbCrLf
            .Parity = IO.Ports.Parity.None
            .StopBits = IO.Ports.StopBits.One
            .Encoding = System.Text.Encoding.GetEncoding("windows-1250")
        End With

    End Sub

    Public Sub Connect()
        'try to open serial port
        'The possible occuring event is not catched to signalize it to the calling process.
        SerialPort.Open()

        If SerialPort.IsOpen Then
            'write one empty line to set device in defined state
            SerialPort.WriteLine("")
            UpdateState()
            SerialPort.WriteLine("Log on")
        End If

    End Sub

    Public Sub Disconnect()
        If SerialPort.IsOpen Then

            'stop logging
            SerialPort.WriteLine("Log off")

            'close port
            SerialPort.Close()

            'setting new lists of modes and units
            'effectivly erasing them because port is closes
            SetNewModes()
            SetNewUnits()

            _AbsMax.Value = String.Empty
            _AbsMax.Unit = String.Empty
            _Filter = 0
            _Maximum.Value = String.Empty
            _Maximum.Unit = String.Empty
            _MeasuredValue.Value = String.Empty
            _MeasuredValue.Unit = String.Empty
            _Minimum.Value = String.Empty
            _Minimum.Unit = String.Empty
            _Mode = ModeState.NotSet
            _NoProbe = False
            _Overload = False
            _Range.Value = String.Empty
            _Range.Unit = String.Empty
            _Reference.Value = String.Empty
            _Reference.Unit = String.Empty
            _Serial = String.Empty
            _Time = 0
            _Unit = UnitState.UnitNotSet
            _Version.Value = String.Empty
            _Version.Major = 0
            _Version.Minor = 0
        End If
    End Sub

    Public Sub RelativeSet(ByVal Reference As Double)
        'use Str() for conversion to get always a "." as decimal separator
        SerialPort.WriteLine("relative " & Str(Reference))
    End Sub

    Public Sub RestoreDefault()
        SerialPort.WriteLine("default")
        UpdateState()
        SerialPort.WriteLine("Log on")
    End Sub

    Public Sub UpdateState()
        SerialPort.WriteLine("status")
    End Sub

    Public Sub ZeroSet()
        SerialPort.WriteLine("zero set")
    End Sub

#End Region

#Region "Propterties"

    Public ReadOnly Property AbsMax As MeasuredValueStruct
        Get
            Return _AbsMax
        End Get
    End Property

    Public Property Coupling As CouplingState
        Get
            Return _Coupling
        End Get
        Set(ByVal Coupling As CouplingState)
            Select Case Coupling
                Case CouplingState.CouplingDC
                    SerialPort.WriteLine("coupling DC")
                Case CouplingState.CouplingAC
                    SerialPort.WriteLine("coupling AC")
            End Select
        End Set
    End Property

    Public Property Digits As Integer
        Get
            Return _Digits
        End Get
        Set(ByVal Digits As Integer)
            SerialPort.WriteLine("digits " & Digits.ToString)
        End Set
    End Property

    Public Property Filter As Integer
        Get
            Return _Filter
        End Get
        Set(ByVal Filter As Integer)
            SerialPort.WriteLine("filter " & Filter.ToString)
        End Set
    End Property

    Public Property Gain As GainState
        Get
            Return _Gain
        End Get
        Set(ByVal Gain As GainState)
            Select Case Gain
                Case GainState.GainX1
                    SerialPort.WriteLine("gain 1")
                    'Changing the gain also changes the range. So we update the range.
                    SerialPort.WriteLine("range")
                Case GainState.GainX10
                    SerialPort.WriteLine("gain 10")
                    'Changing the gain also changes the range. So we update the range.
                    SerialPort.WriteLine("range")
                Case GainState.GainX100
                    SerialPort.WriteLine("gain 100")
                    'Changing the gain also changes the range. So we update the range.
                    SerialPort.WriteLine("range")
            End Select
        End Set
    End Property

    Public ReadOnly Property IsConnected() As Boolean
        Get
            Return SerialPort.IsOpen
        End Get
    End Property

    Public Property KeyLock As KeyState
        Get
            Return _KeyLock
        End Get
        Set(ByVal KeyLock As KeyState)
            Select Case KeyLock
                Case KeyState.KeyboardLocked
                    SerialPort.WriteLine("keys off")
                Case KeyState.KeyboardUnlocked
                    SerialPort.WriteLine("keys on")
            End Select
        End Set
    End Property

    Public ReadOnly Property Maximum As MeasuredValueStruct
        Get
            Return _Maximum
        End Get
    End Property

    Public ReadOnly Property MeasuredValue As MeasuredValueStruct
        Get
            Return _MeasuredValue
        End Get
    End Property

    Public ReadOnly Property Minimum As MeasuredValueStruct
        Get
            Return _Minimum
        End Get
    End Property

    Public Property Mode As ModeState
        Get
            Return _Mode
        End Get
        Set(ByVal Mode As ModeState)
            Select Case Mode
                Case ModeState.Absolute
                    SerialPort.WriteLine("absolute")
                Case ModeState.Relative
                    SerialPort.WriteLine("relative set")
                Case ModeState.Minimum
                    SerialPort.WriteLine("minimum")
                Case ModeState.Maximum
                    SerialPort.WriteLine("maximum")
                Case ModeState.AbsMax
                    SerialPort.WriteLine("amax")
            End Select
        End Set
    End Property

    Public ReadOnly Property Modes As Dictionary(Of ModeState, String)
        Get
            Return _ModeList
        End Get
    End Property

    Public ReadOnly Property NoProbe As Boolean
        Get
            Return _NoProbe
        End Get
    End Property

    Public ReadOnly Property Overload As Boolean
        Get
            Return _Overload
        End Get
    End Property

    Public Property PortName As String
        Get
            Return SerialPort.PortName
        End Get
        Set(ByVal PortName As String)
            SerialPort.PortName = PortName
        End Set
    End Property

    Public ReadOnly Property Range As MeasuredValueStruct
        Get
            Return _Range
        End Get
    End Property

    Public ReadOnly Property Reference As MeasuredValueStruct
        Get
            Return _Reference
        End Get
    End Property

    Public ReadOnly Property Serial As String
        Get
            Return _Serial
        End Get
    End Property

    Public Property Sound As SoundState
        Get
            Return _Sound
        End Get
        Set(ByVal Sound As SoundState)
            Select Case Sound
                Case SoundState.SoundOff
                    SerialPort.WriteLine("sound off")
                Case SoundState.SoundOn
                    SerialPort.WriteLine("sound on")
            End Select
        End Set
    End Property

    Public Property Time As Integer
        Get
            Return _Time
        End Get
        Set(ByVal Time As Integer)
            SerialPort.WriteLine("time " & Time.ToString)
        End Set
    End Property

    Public Property Unit As UnitState
        Get
            Return _Unit
        End Get
        Set(ByVal Unit As UnitState)
            Select Case Unit
                Case UnitState.UnitTesla
                    SerialPort.WriteLine("unit T")
                Case UnitState.UnitGauss
                    SerialPort.WriteLine("unit G")
                Case UnitState.UnitOersted
                    SerialPort.WriteLine("unit Oe")
                Case UnitState.UnitA_per_m
                    SerialPort.WriteLine("unit A/m")
                Case UnitState.UnitA_per_cm
                    SerialPort.WriteLine("unit A/cm")
            End Select
            'Changing the unit also changes the range. So we update the range.
            SerialPort.WriteLine("range")
        End Set
    End Property

    Public ReadOnly Property Units As Dictionary(Of UnitState, String)
        Get
            Return _UnitLIst
        End Get
    End Property

    Public ReadOnly Property Version As String
        Get
            Return _Version.Value
        End Get
    End Property

    Public Property Zero As Integer
        Get
            Return _Zero
        End Get
        Set(ByVal Zero As Integer)
            SerialPort.WriteLine("zero " & Zero.ToString)
        End Set
    End Property

#End Region

    Private Sub SerialPort_DataReceivedSync(ByVal e As DataReceivedSyncEventArgs) Handles SerialPort.DataReceivedSync
        ParseString(e.ReceivedData)
    End Sub

    Private Sub ParseString(ByVal Data As String)
        Dim Buffer As String
        Dim Buffers As String()
        Dim Value As MeasuredValueStruct

        Dim ValueRegEx As New Regex("^-?\d+\.\d+ (M|k|m|µ|n)?(T|Gs|Oe|A/m|A/cm)", RegexOptions.Compiled)
        Dim ValueMatch As Match = ValueRegEx.Match(Data)
        Dim SecondValue As String = ValueRegEx.Replace(Data, String.Empty)

        'a line which starts with a measured value was detected
        If ValueMatch.Success Then
            Buffers = ValueMatch.Value.Split(" ")
            _MeasuredValue.Value = Buffers(0)
            _MeasuredValue.Unit = Buffers(1)
            'Overload = False because a new measured value has been received
            _Overload = False
            'probe is present because a new measured value has been received
            ResetNoProbe()
            RaiseEvent NewMeasuredValue(Me, New EventArgs)
            If SecondValue.StartsWith(", ref ") Then
                SetNewMode(ModeState.Relative)
                Buffers = SecondValue.Substring(6).Split(" ")
                Value.Value = Buffers(0)
                Value.Unit = Buffers(1)
                SetNewReference(Value)
            ElseIf SecondValue.StartsWith(", min ") Then
                SetNewMode(ModeState.Minimum)
                Buffers = SecondValue.Substring(6).Split(" ")
                Value.Value = Buffers(0)
                Value.Unit = Buffers(1)
                SetNewMinimum(Value)
            ElseIf SecondValue.StartsWith(", max ") Then
                SetNewMode(ModeState.Maximum)
                Buffers = SecondValue.Substring(6).Split(" ")
                Value.Value = Buffers(0)
                Value.Unit = Buffers(1)
                SetNewMaximum(Value)
            ElseIf SecondValue.StartsWith(", |m| ") Then
                SetNewMode(ModeState.AbsMax)
                Buffers = SecondValue.Substring(6).Split(" ")
                Value.Value = Buffers(0)
                Value.Unit = Buffers(1)
                SetNewAbsMax(Value)
            Else
                SetNewMode(ModeState.Absolute)
                Value.Value = String.Empty
                Value.Unit = String.Empty
                _Maximum = Value
                _Minimum = Value
                _Reference = Value
            End If
        ElseIf Data.StartsWith("overload") Then
            _Overload = True
            RaiseEvent NewOverload(Me, New EventArgs)
        ElseIf Data = "no probe" Then
            SetNewNoProbe()
        ElseIf Data.StartsWith("coupling is ") Then
            Buffer = Data.Substring(12)
            Select Case Buffer
                Case "DC"
                    _Coupling = CouplingState.CouplingDC
                    RaiseEvent NewCoupling(Me, New EventArgs)
                Case "AC"
                    _Coupling = CouplingState.CouplingAC
                    RaiseEvent NewCoupling(Me, New EventArgs)
            End Select
        ElseIf Data.StartsWith("filter is ") Then
            _Filter = Val(Data.Substring(10))
            RaiseEvent NewFilter(Me, New System.EventArgs)
        ElseIf Data.StartsWith("analog gain is ") Then
            Buffer = Data.Substring(15)
            Select Case Buffer
                Case "x1"
                    _Gain = GainState.GainX1
                    RaiseEvent NewGain(Me, New EventArgs)
                Case "x10"
                    _Gain = GainState.GainX10
                    RaiseEvent NewGain(Me, New EventArgs)
                Case "x100"
                    _Gain = GainState.GainX100
                    RaiseEvent NewGain(Me, New EventArgs)
            End Select
        ElseIf Data.StartsWith("keys are ") Then
            Buffer = Data.Substring(9)
            Select Case Buffer
                Case "unlocked"
                    _KeyLock = KeyState.KeyboardUnlocked
                    RaiseEvent NewKeyLock(Me, New EventArgs)
                Case "locked"
                    _KeyLock = KeyState.KeyboardLocked
                    RaiseEvent NewKeyLock(Me, New EventArgs)
            End Select
        ElseIf Data.StartsWith("range is ") Then
            Buffers = Data.Substring(9).Split(" ")
            _Range.Value = Buffers(0)
            _Range.Unit = Buffers(1)
            RaiseEvent NewRange(Me, New System.EventArgs)
        ElseIf Data.StartsWith("serial no. is ") Then
            SetNewSerial(Data.Substring(14))
        ElseIf Data.StartsWith("sound is ") Then
            Buffer = Data.Substring(9)
            Select Case Buffer
                Case "on"
                    _Sound = SoundState.SoundOn
                    RaiseEvent NewSound(Me, New EventArgs)
                Case "off"
                    _Sound = SoundState.SoundOff
                    RaiseEvent NewSound(Me, New EventArgs)
            End Select
        ElseIf Data.StartsWith("integration time is ") Then
            _Time = Val(Data.Substring(20))
            RaiseEvent NewTime(Me, New System.EventArgs)
        ElseIf Data.StartsWith("unit is ") Then
            Buffer = Data.Substring(8)
            Select Case Buffer
                Case "T"
                    SetNewUnit(UnitState.UnitTesla)
                Case "Gs"
                    SetNewUnit(UnitState.UnitGauss)
                Case "Oe"
                    SetNewUnit(UnitState.UnitOersted)
                Case "A/m"
                    SetNewUnit(UnitState.UnitA_per_m)
                Case "A/cm"
                    SetNewUnit(UnitState.UnitA_per_cm)
            End Select
        ElseIf Data.StartsWith("firmware version is ") Then
            SetNewVersion(Data.Substring(20))
        ElseIf Data.StartsWith("zero compensation value is ") Then
            _Zero = Val(Data.Substring(27))
            RaiseEvent NewZero(Me, New System.EventArgs)
        ElseIf Data.StartsWith("offset out of range, hence zero compensation value is ") Then
            _Zero = Val(Data.Substring(54))
            RaiseEvent ZeroOutOfRange(Me, New System.EventArgs)
            RaiseEvent NewZero(Me, New System.EventArgs)
        End If
    End Sub

#Region "SetNew_Property methods"

    Private Sub SetNewAbsMax(ByVal AbsMax As MeasuredValueStruct)
        If (_AbsMax.Value <> AbsMax.Value) Or (_AbsMax.Unit <> AbsMax.Unit) Then
            _AbsMax = AbsMax
            RaiseEvent NewAbsMax(Me, New System.EventArgs)
        End If
    End Sub

    Private Sub SetNewMaximum(ByVal Maximum As MeasuredValueStruct)
        If (_Maximum.Value <> Maximum.Value) Or (_Maximum.Unit <> Maximum.Unit) Then
            _Maximum = Maximum
            RaiseEvent NewMaximum(Me, New System.EventArgs)
        End If
    End Sub

    Private Sub SetNewMinimum(ByVal Minimum As MeasuredValueStruct)
        If (_Minimum.Value <> Minimum.Value) Or (_Minimum.Unit <> Minimum.Unit) Then
            _Minimum = Minimum
            RaiseEvent NewMinimum(Me, New System.EventArgs)
        End If
    End Sub

    Private Sub SetNewMode(ByVal Mode As ModeState)
        If _Mode <> Mode Then
            _Mode = Mode
            RaiseEvent NewMode(Me, New System.EventArgs)
        End If
    End Sub

    Private Sub SetNewModes()
        _ModeList.Clear()
        If (SerialPort.IsOpen) And (_NoProbe = False) Then
            _ModeList.Add(ModeState.Absolute, "absolute")
            _ModeList.Add(ModeState.Relative, "relative")
            _ModeList.Add(ModeState.Minimum, "Minimum")
            _ModeList.Add(ModeState.Maximum, "Maximum")
            ' add abs. max only if software version is >= 1.4 as it is the first version that support abs. max
            If (_Version.Major > 1) Or (_Version.Major = 1 And _Version.Minor >= 4) Then
                _ModeList.Add(ModeState.AbsMax, "abs. Max")
            End If
        End If
        RaiseEvent NewModes(Me, New System.EventArgs)
    End Sub

    Private Sub SetNewReference(ByVal Reference As MeasuredValueStruct)
        If (_Reference.Value <> Reference.Value) Or (_Reference.Unit <> Reference.Unit) Then
            _Reference = Reference
            RaiseEvent NewReference(Me, New System.EventArgs)
        End If
    End Sub

    Private Sub SetNewNoProbe()
        If _NoProbe = False Then
            _NoProbe = True
            SetNewModes()
            SetNewUnits()
            RaiseEvent NewNoProbe(Me, New System.EventArgs)
        End If
    End Sub

    Private Sub ResetNoProbe()
        'if the state no probe ends, a new probe has been connected to the FM302
        If _NoProbe = True Then
            _NoProbe = False
            'update state to reflect settings of new probe
            UpdateState()
        End If
    End Sub

    Private Sub SetNewSerial(ByVal Serial As String)
        _Serial = Serial

        ' units depend on hardware distinguishable from serial so update them
        SetNewUnits()

        RaiseEvent NewSerial(Me, New System.EventArgs)
    End Sub

    Private Sub SetNewUnit(Unit As UnitState)
        If (SerialPort.IsOpen) And (_NoProbe = False) Then
            If _Unit <> Unit Then
                _Unit = Unit
                RaiseEvent NewUnit(Me, New System.EventArgs)
            End If
        Else
            _Unit = UnitState.UnitNotSet
        End If
    End Sub

    Private Sub SetNewUnits()
        Dim VersionBatch As Integer

        _UnitLIst.Clear()
        If (SerialPort.IsOpen) And (_NoProbe = False) Then
            Dim ValueRegEx As New Regex("^(\d{4})827\d+", RegexOptions.Compiled)
            Dim ValueMatch As Match = ValueRegEx.Match(_Serial)
            If ValueMatch.Success Then
                VersionBatch = CInt(ValueMatch.Result("$1"))
            End If

            _UnitLIst.Add(UnitState.UnitTesla, "Tesla")
            _UnitLIst.Add(UnitState.UnitGauss, "Gauss")
            _UnitLIst.Add(UnitState.UnitOersted, "Oersted")
            _UnitLIst.Add(UnitState.UnitA_per_m, "A/m")
            ' add A/cm only if device is from 2018-08 or newer and software version is >= 1.4
            ' as version 1.4 is the first version that supports A/cm
            ' and a hardware first shipped with batch 1808827... has required prerequisites
            If (VersionBatch >= 1808) And ((_Version.Major > 1) Or (_Version.Major = 1 And _Version.Minor >= 4)) Then
                _UnitLIst.Add(UnitState.UnitA_per_cm, "A/cm")
            End If
        End If
        RaiseEvent NewUnits(Me, New System.EventArgs)

    End Sub

    Private Sub SetNewVersion(ByVal Version As String)
        Dim Buffers As String()

        Dim ValueRegEx As New Regex("^\d+\.\d+", RegexOptions.Compiled)
        'first char of version is "v", so discard and use rest of string -> Substring(1)
        Dim ValueMatch As Match = ValueRegEx.Match(Version.Substring(1))

        Buffers = ValueMatch.Value.Split(".")
        _Version.Major = CInt(Buffers(0))
        _Version.Minor = CInt(Buffers(1))
        _Version.Value = Version

        ' modes and units depend on software version so update them
        SetNewModes()
        SetNewUnits()

        RaiseEvent NewVersion(Me, New System.EventArgs)
    End Sub

#End Region

End Class
